unit Unit1;

{$mode objfpc}{$H+}

  /////////////////////////
 // Robert Rozee, 2024 ///
/////////////////////////

interface

uses
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, StdCtrls, ExtCtrls,
  Menus, BaseUnix, TermIO, Linux, Types, Math, LCLType, Clipbrd, Unix;

///////////////////////////////////////////////////////////////////////////////////
// remember to add -dUseCThreads to Project -> Project Options -> Custom Options //
///////////////////////////////////////////////////////////////////////////////////

// for use of inotify routines, see:
// https://www.freepascal.org/~michael/articles/dirwatch/dirwatch.pdf

type

  { TForm1 }

  TForm1 = class(TForm)
    Panel1: TPanel;
    Label01: TLabel;
    Label02: TLabel;
    Label03: TLabel;
    Label04: TLabel;
    Label12: TLabel;
    Label23: TLabel;
    Label34: TLabel;
    LabelUP: TLabel;
    LabelDN: TLabel;
    LabelFL: TLabel;
    Timer1: TTimer;
    PopupMenu1: TPopupMenu;
    MenuItem1: TMenuItem;
    MenuItem2: TMenuItem;
    MenuItem3: TMenuItem;
    procedure FormActivate(Sender: TObject);
    procedure FormClose(Sender: TObject; var {%H-}CloseAction: TCloseAction);
    procedure FormKeyDown(Sender: TObject; var Key: Word; {%H-}Shift: TShiftState);
    procedure FormMouseWheel(Sender: TObject; {%H-}Shift: TShiftState;
      WheelDelta: Integer; {%H-}MousePos: TPoint; var {%H-}Handled: Boolean);
    procedure MouseClickEvent(Sender: TObject);
    procedure MenuItem1Click(Sender: TObject);
    procedure MenuItem2Click(Sender: TObject);
    procedure MenuItem3Click(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
  private

  public

  end;

var
  Form1: TForm1;

implementation

{$R *.lfm}

{ TForm1 }

////////////////////////////////////////////////////////////////////////////////
// 1. use Audacity to convert sound sources to RAW format:
// Signed, 16-bit, PCM, Mono @22050Hz
//
// 2. use bin2obj to convert RAW files into include files:
// bin2obj -o sound1.inc -c plugin Insert.raw
// bin2obj -o sound2.inc -c unplug Remove.raw
////////////////////////////////////////////////////////////////////////////////

{$I sound1.inc}
{$I sound2.inc}
{$I sound.inc}

const CLlength=20;
var ChangeList:array[0..CLlength-1] of record
                                         name:string;
                                         mask:integer;
                                         time:int64
                                       end;
      PortList:TStringList;
            R4:array [1..4] of record min, max, hgt:integer end;
            R3:array [1..3] of record min, max, hgt:integer end;


const PLUGINflag:boolean=false;                // flag to indicate a new device has been plugged in
      UNPLUGflag:boolean=false;                // flag to indicate an existing device has been unplugged
      UpdateFlag:boolean=false;                // flag to indicate that the display needs updating
      ListOffset:integer=MaxInt div 2;
          clHead:integer=0;                   // change list head pointer
          clTail:integer=0;                   // change list tail pointer
       FlashLine:integer=0;                   // shows which line in the display should be "flashed"

 type TCheckThread = class(TThread)           // thread that monitors /dev directory for any changes
     private
     protected
       procedure Execute; override;
     end;

 type TSoundThread = class(TThread)           // thread that plays "plugin" and "unplug" sounds
      private
      protected
        procedure Execute; override;
      end;


// separate thread used to check for changes in /dev
// pretty much identical to Michael Van Canneyt's example code here:
// https://www.freepascal.org/~michael/articles/dirwatch/dirwatch.pdf
procedure TCheckThread.Execute;
var fd, wd, bytesread, namelength:integer;
                         filename:string;
                           buffer:array [0..4095] of byte;
                            event:^inotify_event;
                              fds:Tfdset;
                                p:pchar;
begin
  fd:=inotify_init();
  try
    wd:=inotify_add_watch(fd, '/dev', IN_CREATE or IN_DELETE);         // wd never used, as we are only setting ONE watch
    fpFD_Zero(fds);
    fpFD_SET(fd, fds);

    while (fpSelect(fd+1, @fds, nil, nil, nil)>=0) do
    begin
      bytesread:=fpRead(fd, buffer, sizeof(buffer));
      event:=@buffer;

      While ((pchar(event)-@buffer)<bytesread) do
      begin
        namelength:=event^.len;

        if (namelength>0) then
        begin
          p:=@event^.name+namelength-1;
          While (p^=#0) do
          begin
            dec(p);
            dec(namelength)
          end
        end;

        setlength(filename, namelength);
        if (namelength>0) then move(event^.name,filename[1],namelength);
                               //     source    destination   size

//      if pos('tty',filename)=1 then                                  // commented out to allow for "rfcomm0", etc
//      if event^.wd=wd then                                           // check only needed if multiple events registered
        with ChangeList[clHead] do                                     // (potential) port names are saved in a circular
        begin                                                          // queue, ready to be picked up by 20ms timer
          name:=filename;
          mask:=event^.mask;                                           // either IN_CREATE or IN_DELETE
          time:=GetTickCount64;                                        // time at which plugin/remove event happened;
          if (event^.mask and IN_CREATE)>0 then inc(time, 100);        // if it was a plugin event then add 100ms
          clHead:=(clHead+1) mod CLlength
        end;                                                           // finally, make new entry available

//      writeln('Change 0x', IntToHex(event^.mask, 8),
//              ' detected for "/dev/', filename, '"');
        ptruint(event):=ptruint(event)+sizeof(inotify_event)+event^.len-1
      end
    end;
  finally
    fpClose(fd)
  end
end;


procedure TSoundThread.Execute;                // SOUND THREAD - as a temporary solution uses "aplay"
begin
  repeat
    if PLUGINflag then begin
                         PLUGINflag:=false;
                         APlaySound(1)
                      end;
    if UNPLUGflag then begin
                         UNPLUGflag:=false;
                         APlaySound(2)
                      end;
    sleep(100)
  until false
end;

{
function normalize(S:string):string;           // expands any symlink at the end of S
var n, I:integer;
       T:string;
begin
  T:=fpReadLink(S);                            // try to follow a symlink (returns '' if fails)

  if T='' then result:=S else                  // if T is empty, there was no symlink to follow
  begin
    n:=0;
    while pos('../', T)=1 do                   // strip off as many leading "../" from T as possible
    begin
      delete(T,1,3);
      inc(n)                                   // keep count of number of steps involved
    end;

    inc(n);                                    // +1 to also remove the (old) filename at end of S

    while n>0 do                               // strip off (old) filename and directories from end of S
    begin
      I:=length(S)-1;                          // new length for S with the last character removed
      setlength(S, I);                         // we trim one character at a time...
      if (I=0) or (S[I]='/') then dec(n)       // and count down for each "/" found; falls through if S=''
    end;                                       // when the loop exits it should leave a trailing "/"

    T:=S+T;                                    // concatenate the two trimmed strings
    repeat
      I:=pos('//', T);                         // fixup for any double "//" where S and T are joined
      if I<>0 then delete(T, I, 1)             // (in the present application this should never happen)
    until I=0;
    result:=T                                  // return an absolute path
  end
end;
}
{
type
   TSerialStruct = packed record
          typ: cint;
          line: cint;
          port: cuint;
          irq: cint;
          flags: cint;
          xmit_fifo_size: cint;
          custom_divisor: cint;
          baud_base: cint;
          close_delay: cushort;
          io_type: cchar;
          reserved_char: pcchar;
          hub6: cint;
          closing_wait: cushort; // time to wait before closing
          closing_wait2: cushort; // no longer used...
          iomem_base: pcchar;
          iomem_reg_shift: cushort;
          port_high: clong; // cookie passed into ioremap
          overrun: array [1..64] of byte
   end;
}

// initial method was based on information obtained from the following two sites:
// https://www.lazarusforum.de/viewtopic.php?p=72837
// https://stackoverflow.com/questions/2530096
// updated when kernel 6.8 changed the name of the 8250 driver - grrr!!!
// now also accepts '/sys/bus/serial-base/' as indicator of a fixed UART
////////////////////////////////////////////////////////////////////////////////
// addendum: better fix - check for exists "/sys/class/tty/<DeviceName>/type" //
////////////////////////////////////////////////////////////////////////////////
function CheckDevice(DeviceName:string):boolean;               // checks to see if the device named is a live
var {DriverPath,} S:string;                                    // ... serial port. this is done by checking
               FD:longint;                                     // ... entries in /sys/class/tty including the
//             SS:TSerialStruct;                               // ... presence of a link to a device driver.
                T:text;
                I:integer;
             tios:TermIOS;
begin
  Result:=false;

  if (DeviceName<>'.') and (DeviceName<>'..') then
///////////////////////////////////////////////////////////////////////////////// vvvvvvv                 vv vvvvv  vv
  if fpAccess('/dev/'+DeviceName, R_OK+W_OK)=0 then                            // exclude devices we have NO ACCESS TO
///////////////////////////////////////////////////////////////////////////////// ^^^^^^^                 ^^ ^^^^^^ ^^
//if FileExists('/sys/class/tty/'+DeviceName+'/device/driver')  then           // this suffices with FPC prior to 3.20
  if DirectoryExists('/sys/class/tty/'+DeviceName+'/device/driver')  then      // from FPC 3.20 onwards we need this instead
  begin

{
    S:='/sys';
    S:=normalize(S+'/class');                                  // need all this palaver just so we can
    S:=normalize(S+'/tty');                                    // check that DriverPath does not begin
    S:=normalize(S+'/'+DeviceName);                            // with "/sys/bus/serial-base", as this
    S:=normalize(S+'/device');                                 // would indicate a real ttySxx under
    S:=normalize(S+'/driver');                                 // kernel 6.8+

    DriverPath:=S;

    if (pos('/sys/bus/serial-base/', DriverPath)<>1) and                       // kernel 6.8+ check for a removable (USB) device
       (ExtractFileName(DriverPath)<>'serial8250') then Result:=true else      // ... pre-6.8 check for a removable (USB) device
}

/////////////////////////////////////////////////////////////////////////////////////    #  #  #
    if not FileExists('/sys/class/tty/'+DeviceName+'/type') then Result:=true else //     \ | /
/////////////////////////////////////////////////////////////////////////////////////      \|/
// the above is an alternative to checking the driver name against "serial8250" /////    #--#--#
// and just needs the directory containing the /type file to be searchable;     /////      /|\
// it does NOT need the /type file itself to be readable                        /////     / | \
/////////////////////////////////////////////////////////////////////////////////////    #  #  #

    begin                                                      // we ONLY do the following on FIXED serial ports
      S:='/sys/class/tty/'+DeviceName+'/type';
      try                                                      // beware: read access may be blocked with pre-6.8 kernels
        I:=-1;
        assign(T, S);
        reset(T);
        if not eof(T) then readln(T, I);
        close(T)
      except end;                                              // note: reset, readln, or close could raise an exception

      I:=-1;

      if I<0 then                                              // we run the next test if we could NOT read the "type" file
      try
        FD:=fpOpen('/dev/'+DeviceName, O_RDWR or O_NONBLOCK or O_NOCTTY);
        if FD>0 then
        begin
//        if fpIOCtl(FD, TIOCGSERIAL, @SS)<>-1 then I:=SS.typ  // try to use ioctl call to retrieve port's "type"
          if fpIOCtl(FD, TCGETS, @tios)=0 then I:=255;         // better??? alternative, same as IsATTY(FD)
//        if IsATTY(FD)<>0) then I:=255                        // this would work just as well as ioctl/TCGETS
          fpClose(FD)
        end
      except end;                                              // try-except-end probably not needed, but kept just in case

      Result:=(I>0)                                            // port exist if "type" is 1, 2, 3, 4...255
    end
  end
end;


// does a full scan of /sys/class/tty and checks every current entry (using CheckDevice) to create an initial list of live serial ports
procedure PopulatePortList;
var S1,S2:string;
    I,J,K:integer;
     flag:boolean;
       SR:TSearchRec;
begin
  PortList.Clear;

  if FindFirst('/sys/class/tty/*', faDirectory , SR) = 0 then          // initially we scan /sys/class/tty for potential serial ports
  repeat                                                               //                   ~~~~~~~~~~~~~~
    if CheckDevice(SR.Name) then PortList.Append(SR.Name)              // check that each port found is live, add to memo if it is
  until FindNext(SR) <> 0;
  FindClose(SR);

  flag:=true;
  while (PortList.Count>1) and flag do                                 // excessively complicated sort routine, tries to ensure that the
  begin                                                                // 'fixed' serial ports appear first, and that port numbers are
    flag:=false;                                                       // ordered correctly: 0,1,2...,8,9,10,11, etc.
    for I:=0 to PortList.Count-2 do
    begin
      S1:=PortList.Strings[I];
      J:=1+length(S1);

      S2:=PortList.Strings[I+1];
      K:=1+length(S2);

      if (J-K)<0 then begin                                    // pack S1 with zeros to left of numeric part
                        while (J>1) and (S1[J-1] in ['0'..'9']) do dec(J);
                        while length(S1)<length(S2) do insert('0', S1, J)
                      end else
      if (K-J)<0 then begin                                    // pack S2 with zeros to left of numeric part
                        while (K>1) and (S2[K-1] in ['0'..'9']) do dec(K);
                        while length(S2)<length(S1) do insert('0', S2, K)
                      end;

      J:=pos('ttyS',S1);                                       // =1 if is a 'fixed' serial port
      K:=pos('ttyS',S2);                                       // =1 if is a 'fixed' serial port

      if ((J<>1) and (K=1)) or                                 // bubble ttyS* ports to top of the list
         ((J=K) and (S1>S2)) then                              // within respective groups sort alphabetically
      begin
        PortList.Exchange(I, I+1);
        flag:=true                                             // flag set if at least one swap during this pass
      end
    end
  end
end;


// 20ms timer that picks up port names from the queue, checks that they are live, and if so adds them into the list held in PortList.
procedure TForm1.Timer1Timer(Sender: TObject);
const TTS:int64=0;
var DeviceName:String;
       Mask, I:integer;
begin
  while clHead<>clTail do                                      // it is extremely unlikely this will loop through more than once
  begin
    if ChangeList[clTail].time>GetTickCount64 then break;      // if a plugin events is found, queue processing is delayed by 100ms to
                                                               // ... allow permissions to be set. note: queue ORDER is still PRESERVED
    DeviceName:=ChangeList[clTail].name;
    Mask:=ChangeList[clTail].mask;

    I:=PortList.IndexOf(DeviceName);                           // get the index of the port that has changed;
    if I>=0 then PortList.Delete(I);                           // if found, delete the port from the list (irrespective of Mask value)

    case Mask of IN_CREATE:if CheckDevice(DeviceName) then                     // check if it is a 'live' serial port
                           begin
//                           writeln('adding ', DeviceName);
                             PortList.Append(DeviceName);                      // add (live) port to the end of the list, and
                             ListOffset:=MaxInt div 2;                         // ... ensure that the ADDED item is visible
                             Form1.Caption:=UTF8Encode(#$2611)+'  '+DeviceName;        // (box with tick)
                             TTS:=GetTickCount64;                              // active Title-Time-Stamp
                             PLUGINflag:=true                                  // indicate the "plug in" sound should be played
                           end;
                 IN_DELETE:if I>=0 then                                        // only notify if we deleted something from the list
                           begin
//                           writeln('unplug ', DeviceName);
                             Form1.Caption:=UTF8Encode(#$2612)+'  '+DeviceName;       // (box with cross)
                             TTS:=GetTickCount64;                              // active Title-Time-Stamp
                             UNPLUGflag:=true                                  // indicate the "un-plug" sound should be played
                           end
    end;

    clTail:=(clTail+1) mod CLlength;                           // discard processed event
    UpdateFlag:=true                                           // trigger a display update
  end;
                                                               // >>> note: UpdateFlag may be set above, OR by
  if UpdateFlag then                                           // >>> various keyboard, mouse, or menu events.
  begin
    case PortList.Count of 0:begin                             // no ports -> just display a message to this effect
                               Label01.Caption:=' ';
                               Label02.Caption:='no serial';
                               Label03.Caption:='ports found';
                               Label04.Caption:=' ';
                               Label12.Caption:=' ';
                               Label23.Caption:=' ';
                               Label34.Caption:=' ';
                               ListOffset:=0                   // (too few items to be able to scroll display)
                             end;
                           1:begin                             // **ONE** port in list
                               Label01.Caption:=' ';           // because AutoSize is turned on, every Caption
                               Label02.Caption:=' ';           // must contain something - even if it is just
                               Label03.Caption:=' ';           // a single space. without this an empty caption
                               Label04.Caption:=' ';           // will 'roll up' to zero height and disrupt the
                               Label12.Caption:=' ';           // rest of the Form1 layout.
                               Label23.Caption:=PortList[0];
                               Label34.Caption:=' ';
                               ListOffset:=0                   // (too few items to be able to scroll display)
                             end;
                           2:begin                             // **TWO** ports in list
                               Label01.Caption:=' ';
                               Label02.Caption:=PortList[0];
                               Label03.Caption:=PortList[1];
                               Label04.Caption:=' ';
                               Label12.Caption:=' ';
                               Label23.Caption:=' ';
                               Label34.Caption:=' ';
                               ListOffset:=0                   // (too few items to be able to scroll display)
                             end;
                           3:begin                             // **THREE** ports in list
                               Label01.Caption:=' ';
                               Label02.Caption:=' ';
                               Label03.Caption:=' ';
                               Label04.Caption:=' ';
                               Label12.Caption:=PortList[0];
                               Label23.Caption:=PortList[1];
                               Label34.Caption:=PortList[2];
                               ListOffset:=0                   // (too few items to be able to scroll display)
                             end;
                           4:begin                             // **FOUR** ports in list
                               Label01.Caption:=PortList[0];
                               Label02.Caption:=PortList[1];
                               Label03.Caption:=PortList[2];
                               Label04.Caption:=PortList[3];
                               Label12.Caption:=' ';
                               Label23.Caption:=' ';
                               Label34.Caption:=' ';
                               ListOffset:=0                   // (too few items to be able to scroll display)
                             end
                        else begin                             // the following line enforces upper/lower bounds
                               ListOffset:=Min(Max(0, ListOffset), PortList.Count-4);
                               Label01.Caption:=PortList[ListOffset+0];
                               Label02.Caption:=PortList[ListOffset+1];
                               Label03.Caption:=PortList[ListOffset+2];
                               Label04.Caption:=PortList[ListOffset+3];
                               Label12.Caption:=' ';
                               Label23.Caption:=' ';           // with 5 or more items we can now scroll, so
                               Label34.Caption:=' '            // ... we DON'T want to zero ListOffset here
                             end
    end; { of case }

    if ListOffset<>0 then LabelUP.Caption:=UTF8Encode(#$2191)          // more enteries hidden UP above
                     else LabelUP.Caption:=UTF8Encode(#$0020);         // nothing above to display
    if PortList.Count>ListOffset+4
                     then LabelDN.Caption:=UTF8Encode(#$2193)          // more entries hidden DOWN below
                     else LabelDN.Caption:=UTF8Encode(#$0020);         // nothing below to display

    UpdateFlag:=false
  end;

  if FlashLine<>0 then                                         // mouse click -> something was copied to the clipboard
  begin
    if FlashLine<100 then                                      // FIRST time through numbers are 1, 2, 3, 4, 12, 23, 34
    begin
      LabelFL.Left:=Panel1.Left;                               // position LabelFL horizontally, set label's width
      LabelFL.Width:=Panel1.Width;
      LabelFL.SendToBack;                                      // ensure LabelFL is BEHIND all the other labels!
      case FlashLine of 1, 2, 3, 4:begin
                                     LabelFL.Top:=R4[FlashLine].min;           // position Label FL vertically behind Label 1, 2, 3 or 4
                                     LabelFL.Height:=R4[FlashLine].hgt         // set LabelFL height
                                   end;
                        12, 23, 34:begin
                                     I:=FlashLine div 10;
                                     LabelFL.Top:=R3[I].min;                   // position Label FL vertically behind Label 12, 23 or 34
                                     LabelFL.Height:=R3[I].hgt                 // set LabelFL height
                                   end
      end; { of case }

      LabelFL.Show;                            // make LabelFL visible; it will act as a background behind whatever it has been postioned behind
      FlashLine:=100                           // set FlashLine to a suitably high number to ensure we take the ELSE path next time round (in 100ms)
    end else begin LabelFL.Hide; FlashLine:=0 end              // SECOND time through -> all done: hide label and reset FlashLine back to zero
  end;

  if (TTS>0) and (GetTickCount64>(TTS+2000)) then              // ready to restore original title bar (2000ms after changing it)
  begin
    Form1.Caption:='TTY ports :';                              // restore original Form1 caption
    TTS:=0                                                     // deactive Title-Time-Stamp
  end
end;


procedure TForm1.FormActivate(Sender: TObject);
const initial:boolean=true;
begin
// one-time startup code, this is run AFTER the program's window has been created
  if initial then
  begin
    initial:=false;
    PortList:=TStringList.Create;
    PopulatePortList;                                  // create a SORTED list of 'live' serial ports

    Form1.Constraints.MinWidth:=Form1.Width;           // ensure Form1 can NOT be resized
    Form1.Constraints.MaxWidth:=Form1.Width;           // attempt to drag a side will now
    Form1.Constraints.MinHeight:=Form1.Height;         // just move the form instead.
    Form1.Constraints.MaxHeight:=Form1.Height;

    R4[1].min:=Label01.Top;                            // caculate the top of the 4 "EVEN" labels
    R4[2].min:=Label02.Top;
    R4[3].min:=Label03.Top;
    R4[4].min:=Label04.Top;

    R4[1].hgt:=Label01.Height;                         // save the heights of the 4 "EVEN" labels
    R4[2].hgt:=Label02.Height;                         // (used in flashing 'hightlight' bar)
    R4[3].hgt:=Label03.Height;
    R4[4].hgt:=Label04.Height;

    R4[1].max:=R4[1].min+R4[1].hgt-1;                  // caculate the bottom of the 4 "EVEN" labels
    R4[2].max:=R4[2].min+R4[2].hgt-1;
    R4[3].max:=R4[3].min+R4[3].hgt-1;
    R4[4].max:=R4[4].min+R4[4].hgt-1;


    R3[1].min:=R4[1].min+(Label12.Height div 4);       // caculate the 'top of text' for the 3 "ODD" labels
    R3[2].min:=R4[2].min+(Label23.Height div 4);       // (these 3 labels are double-height,
    R3[3].min:=R4[3].min+(Label34.Height div 4);       // with their text vertically centred)

    R3[1].hgt:=Label12.Height div 2;                   // calculate the 'height of text' for the 3 "ODD" labels
    R3[2].hgt:=Label23.Height div 2;                   // (these 3 labels are double-height, so
    R3[3].hgt:=Label34.Height div 2;                   // halve value to get actual text height)

    R3[1].max:=R3[1].min+R3[1].hgt-1;                  // caculate the 'bottom of text' for the 3 "ODD" labels
    R3[2].max:=R3[2].min+R3[2].hgt-1;                  // (so the text fills between the 1/4 and 3/4 marks)
    R3[3].max:=R3[3].min+R3[3].hgt-1;

    ALSAstartup;

    Timer1.Enabled:=true;
    TSoundThread.Create(false);
    TCheckThread.Create(false);                        // hereafter we just need to scan /dev/tty* for changes, with
    UpdateFlag:=true                                   //                                ~~~~~~~~~
  end                                                  // all changes added to the end of the list -> no more sorting
end;

procedure TForm1.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  ALSAshutdown
end;


procedure TForm1.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
var flag:boolean;
begin
  flag:=true;
  case Key of VK_UP:dec(ListOffset);                   // scroll UP display by ONE line
            VK_DOWN:inc(ListOffset);                   // scroll DOWN display by ONE line
            VK_HOME:ListOffset:=0;                     // go to TOP of list
             VK_END:ListOffset:=MaxInt div 2;          // go to BOTTOM of list
           VK_PRIOR:dec(ListOffset, 4);                // jump UP list by 4 lines
            VK_NEXT:inc(ListOffset, 4)                 // jump DOWN list by 4 lines
           else     flag:=false
  end; { of case }                                     // note: ListOffset can be out of bounds, bounds are enforced
  if flag then UpdateFlag:=true
end;


procedure TForm1.FormMouseWheel(Sender: TObject; Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
var flag:boolean;
begin
  flag:=true;
  case sign(WheelDelta) of +1:dec(ListOffset);         // scroll UP display by ONE line
                           -1:inc(ListOffset);         // scroll DOWN display by ONE line
                            0:flag:=false
  end; { of case }
  if flag then UpdateFlag:=true
end;


procedure TForm1.MenuItem1Click(Sender: TObject);
begin
  PortList.Clear;                                      // empty out the list, any new devices plugged in will be added
  UpdateFlag:=true
end;


procedure TForm1.MenuItem2Click(Sender: TObject);
begin
  ListOffset:=MaxInt div 2;                            // go to BOTTOM of list
  PopulatePortList;
  UpdateFlag:=true
end;


procedure TForm1.MenuItem3Click(Sender: TObject);
begin
  Form1.Close                                          // exit program
end;


procedure TForm1.MouseClickEvent(Sender: TObject);     // clicking on a port name will copy it to the clipboard with an EOL added
var P:TPoint;
    I:integer;
begin
  P:=Form1.ScreenToClient(Mouse.CursorPos);                    // convert mouse (screen) coordinates to Form1 coordinates

  case PortList.Count of 0:I:=0;                                               // indicates NO items available to copy to clipboard
                         1:if P.Y in [R3[2].min..R3[2].max] then I:=23;        // we use the same numbering the Labelxx names
                         2:if P.Y in [R4[2].min..R4[2].max] then I:=02 else
                           if P.Y in [R4[3].min..R4[3].max] then I:=03;
                         3:if P.Y in [R3[1].min..R3[1].max] then I:=12 else
                           if P.Y in [R3[2].min..R3[2].max] then I:=23 else
                           if P.Y in [R3[3].min..R3[3].max] then I:=34;        // <--- this semicolon MUST be here to ensure
                      else if P.Y in [R4[1].min..R4[1].max] then I:=01 else    // the next ELSE belongs the CASE, not the IF
                           if P.Y in [R4[2].min..R4[2].max] then I:=02 else
                           if P.Y in [R4[3].min..R4[3].max] then I:=03 else
                           if P.Y in [R4[4].min..R4[4].max] then I:=04
  end; { of case }

  if (I>0) and (P.X in [Panel1.Left..Panel1.Left+Panel1.Width-1]) then         // check of left/right bounds not really needed
  try
    ClipBoard.AsText:='/dev/'+TLabel(Form1.FindComponent(Format('Label%.2D', [I]))).Caption+LineEnding;
    FlashLine:=I                                                               // signals Timer1 to flash the label we clicked on
  except end                                                                   // ignore any ClipBoard or FindComponent errors
end;



end.
